import { httpRouter } from "convex/server";
import { httpAction } from "./_generated/server";
import { internal } from "./_generated/api";

const encoder = new TextEncoder();

async function hmacSha256(
  secret: string,
  data: Uint8Array,
): Promise<string> {
  const key = await crypto.subtle.importKey(
    "raw",
    encoder.encode(secret),
    { name: "HMAC", hash: "SHA-256" },
    false,
    ["sign"],
  );
  const sig = await crypto.subtle.sign("HMAC", key, data);
  return [...new Uint8Array(sig)]
    .map((b) => b.toString(16).padStart(2, "0"))
    .join("");
}

async function sha256Hex(data: Uint8Array): Promise<string> {
  const hash = await crypto.subtle.digest("SHA-256", data);
  return [...new Uint8Array(hash)]
    .map((b) => b.toString(16).padStart(2, "0"))
    .join("");
}

const http = httpRouter();

http.route({
  path: "/sahha-webhook",
  method: "POST",
  handler: httpAction(async (ctx, request) => {
    try {
      const signature = (request.headers.get("x-signature") ?? "").trim();
      const eventType = (request.headers.get("x-event-type") ?? "").trim();
      const externalId = (request.headers.get("x-external-id") ?? "").trim();

      if (!signature) {
        return new Response("Missing X-Signature", { status: 400 });
      }

      const secret = process.env.SAHHA_WEBHOOK_SECRET ?? "";
      if (!secret) {
        return new Response("Missing SAHHA_WEBHOOK_SECRET", { status: 500 });
      }

      const rawBody = new Uint8Array(await request.arrayBuffer());

      const computed = await hmacSha256(secret, rawBody);
      if (computed !== signature) {
        return new Response("Invalid signature", { status: 401 });
      }

      const payload = JSON.parse(new TextDecoder().decode(rawBody));
      const eventHash = await sha256Hex(rawBody);

      await ctx.runMutation(internal.webhooks.store, {
        eventType: eventType || "unknown",
        externalId,
        payload,
        eventHash,
      });

      return new Response("ok", { status: 200 });
    } catch (err) {
      console.error("Webhook error:", err);
      return new Response("Server error", { status: 500 });
    }
  }),
});

export default http;
