import express from "express";
import getRawBody from "raw-body";
import crypto from "crypto";
import dotenv from "dotenv";

dotenv.config();

const app = express();
const PORT = process.env.PORT || 3000;
const SAHHA_SECRET = process.env.SAHHA_WEBHOOK_SECRET;

if (!SAHHA_SECRET) {
  console.error("❌ Missing SAHHA_WEBHOOK_SECRET in .env");
  process.exit(1);
}

// IMPORTANT: Do NOT use express.json() or body-parser on this route.
// We must verify the signature against the raw request body.
app.post("/sahha-webhook", async (req, res) => {
  try {
    const signature = (req.header("X-Signature") || "").trim();
    const eventType = (req.header("X-Event-Type") || "").trim();
    const externalId = (req.header("X-External-Id") || "").trim();

    if (!signature) {
      return res.status(400).send("Missing X-Signature");
    }

    const rawBody = await getRawBody(req);

    const computed = crypto
      .createHmac("sha256", SAHHA_SECRET)
      .update(rawBody)
      .digest("hex");

    if (computed !== signature) {
      return res.status(401).send("Invalid signature");
    }

    const payload = JSON.parse(rawBody.toString("utf8"));

    console.log("===== Sahha Webhook Received =====");
    console.log("Event Type:", eventType || "(missing)");
    console.log("External ID:", externalId || "(missing)");
    console.log("Payload:", payload);
    console.log("==================================");

    return res.status(200).send("ok");
  } catch (err) {
    console.error("Webhook error:", err);
    return res.status(500).send("server error");
  }
});

app.listen(PORT, () => {
  console.log(`Listening on http://localhost:${PORT}`);
});
