import { onRequest } from "firebase-functions/v2/https";
import { initializeApp } from "firebase-admin/app";
import { getFirestore } from "firebase-admin/firestore";
import crypto from "crypto";
import getRawBody from "raw-body";

initializeApp();
const db = getFirestore();

export const sahhaWebhook = onRequest(async (req, res) => {
  try {
    const signature = (req.header("X-Signature") || "").trim();
    const eventType = (req.header("X-Event-Type") || "").trim();
    const externalId = (req.header("X-External-Id") || "").trim();

    if (!signature) {
      res.status(400).send("Missing X-Signature");
      return;
    }

    const rawBody = await getRawBody(req);

    const secret = process.env.SAHHA_WEBHOOK_SECRET;
    if (!secret) {
      res.status(500).send("Missing SAHHA_WEBHOOK_SECRET");
      return;
    }

    const computed = crypto
      .createHmac("sha256", secret)
      .update(rawBody)
      .digest("hex");

    if (computed !== signature) {
      res.status(401).send("Invalid signature");
      return;
    }

    const payload = JSON.parse(rawBody.toString("utf8"));
    const eventHash = crypto.createHash("sha256").update(rawBody).digest("hex");

    await db
      .collection("sahha_events")
      .doc(eventType || "unknown")
      .collection("events")
      .doc(eventHash)
      .set({
        eventType,
        externalId,
        payload,
        receivedAt: new Date().toISOString(),
        eventHash,
      });

    res.status(200).send("ok");
  } catch (err) {
    console.error(err);
    res.status(500).send("server error");
  }
});
