import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

const SECRET = Deno.env.get("SAHHA_WEBHOOK_SECRET") ?? "";
const SUPABASE_URL = Deno.env.get("SUPABASE_URL") ?? "";
const SUPABASE_SERVICE_ROLE_KEY =
  Deno.env.get("SUPABASE_SERVICE_ROLE_KEY") ?? "";

const encoder = new TextEncoder();

async function hmacSha256(secret: string, data: Uint8Array): Promise<string> {
  const key = await crypto.subtle.importKey(
    "raw",
    encoder.encode(secret),
    { name: "HMAC", hash: "SHA-256" },
    false,
    ["sign"],
  );
  const sig = await crypto.subtle.sign("HMAC", key, data);
  return [...new Uint8Array(sig)]
    .map((b) => b.toString(16).padStart(2, "0"))
    .join("");
}

async function sha256Hex(data: Uint8Array): Promise<string> {
  const hash = await crypto.subtle.digest("SHA-256", data);
  return [...new Uint8Array(hash)]
    .map((b) => b.toString(16).padStart(2, "0"))
    .join("");
}

Deno.serve(async (req) => {
  try {
    const signature = (req.headers.get("x-signature") ?? "").trim();
    const eventType = (req.headers.get("x-event-type") ?? "").trim();
    const externalId = (req.headers.get("x-external-id") ?? "").trim();

    if (!signature) {
      return new Response("Missing X-Signature", { status: 400 });
    }

    const rawBody = new Uint8Array(await req.arrayBuffer());

    if (!SECRET) {
      return new Response("Missing SAHHA_WEBHOOK_SECRET", { status: 500 });
    }

    const computed = await hmacSha256(SECRET, rawBody);
    if (computed !== signature) {
      return new Response("Invalid signature", { status: 401 });
    }

    const payload = JSON.parse(new TextDecoder().decode(rawBody));
    const eventHash = await sha256Hex(rawBody);

    const supabase = createClient(SUPABASE_URL, SUPABASE_SERVICE_ROLE_KEY);

    const { error } = await supabase.from("sahha_events").upsert(
      {
        event_type: eventType || "unknown",
        external_id: externalId,
        payload,
        event_hash: eventHash,
      },
      { onConflict: "event_hash" },
    );

    if (error) {
      console.error("Insert error:", error);
      return new Response("Database error", { status: 500 });
    }

    return new Response("ok", { status: 200 });
  } catch (err) {
    console.error(err);
    return new Response("Server error", { status: 500 });
  }
});
